#!/usr/bin/env python
# SPDX-FileCopyrightText: 2017-2023 Univention GmbH
# SPDX-License-Identifier: AGPL-3.0-only

from __future__ import print_function

import os
import re
import sys
try:
    from typing import Dict, Iterable, List, Tuple  # noqa: F401
except ImportError:
    pass

from ucr import UCR


# 1. Use descriptive names. What is "srvrlmtch" going to tell anybody?
# 2. If you cannot find a descriptive name it indicated that it's hard to tell what the given object is, i.e. it indicates that one should re-think if the conglomerate of information stored in the object makes sense.
# 3. Object to store Server-Role Information is not needed because role is constant in usecase
# 4. Keep it simple: Objects are for a) encapsulation b) combining methods with scoped data. The SRM class doesn't fit these criteria: you access srm.packages.update() so you don't need srm.count()
# 5. Don't introduce new data for no reason, e.g. pkgrole instead of simply using pkgname
# 6. Don't do work twice: you apply a regexp, so let it extract the relevant parts from the line
# 7. Make a line to one thing, avoid nested constucts like: function_call(data.subobject[index])

# -----

def package_is_ok_on_server_role(pkgname, server_role):  # type: (str, str) -> bool
    return (pkgname, server_role) in {
        ("univention-server-master", "domaincontroller_master"),
        ("univention-server-backup", "domaincontroller_backup"),
        ("univention-server-slave", "domaincontroller_slave"),
        ("univention-server-member", "memberserver"),
    }


def check_server_pkgs(matches, server_role):  # type: (Iterable[Tuple[str, str]], str) -> List[Dict[str, bool]]
    return [
        {pkgname: package_is_ok_on_server_role(pkgname, server_role)}
        for (pkgstat, pkgname) in matches
        if pkgstat.startswith('install') and pkgname != "univention-server-overview"
    ]


# load ucr variables
usi_path = sys.argv[1]
ucr = UCR(usi_path)

# load package stati from file
dpkglist_path = os.path.join(usi_path, 'info', 'dpkg-l')
with open(dpkglist_path) as infofile:
    dpkglist = infofile.read()

# only check relevant packages if they (it) matches server/role
server_role = ucr.get('server/role')
RE_ROLE_PACKAGES_IN_DPKGLIST = re.compile(r"(.*)\t(univention-server-.*|univention-basesystem)\t.*")
metapackage_check_result = check_server_pkgs(RE_ROLE_PACKAGES_IN_DPKGLIST.findall(dpkglist), server_role)

# print check result
if not metapackage_check_result:
    print("ERROR: No server package found!")
    sys.exit(5)
elif len(metapackage_check_result) > 1:
    print("ERROR: more than 1 server package found! %s" % (metapackage_check_result,))
    sys.exit(3)
elif not all(metapackage_check_result[0].values()):
    print("Warning: server/role (%s) does NOT match installed Package" % (server_role,))
    sys.exit(1)
else:
    print("Info: server/role (%s) does match installed Package" % (server_role,))
    sys.exit(0)
